<?php
/**
 * Active Directory Sync Class
 * 
 * This class is used to abstract out the functionallity
 * of connecting to the Active Directory server from
 * the mw2cli agent when requesting lists of email addresses
 * back from a customers Active Directory Installation.
 * 
 * Created Date: 6th July 2009
 * 
 * @author Brian Drysdale <brian.drysdale@ieinternet.com>
 * @copyright IE Internet.com Limited, All rights reserved.
 */
 

class AdSync {
	/**
	 * Server Name/IP
	 * @var string
	 */
	var $server=null;
	/**
	 * Server Port
	 * @var int
	 */
	var $port=null;
	/**
	 * Username
	 * @var string
	 */
	var $user=null;
	/**
	 * Password
	 * @var string
	 */
	var $pass=null;
	/**
	 * Search Domain
	 * @var string
	 */
	var $dn=null;
	/**
	 * Use SSL or not
	 * @var boolean
	 */
	var $ssl=false;
	/**
	 * Use TLS or not
	 * @var boolean
	 */
	var $tls=false;
	/**
	 * LDAP Connection reference
	 * @var resource
	 */
	var $ad = null;
	/**
	 * Search string to use in AD
	 * @var string
	 */
	var $searchString='(&(|(&(objectCategory=group)(objectClass=Group))(&(objectCategory=person)(objectClass=User)))(|(mail=*)(proxyaddresses=smtp*))(!(cn=SystemMailbox*))(!(cn=Administratorek)))';
	/**
	 * Fields to return from AD
	 * @var array
	 */
	var $returnFields = array('proxyaddresses');
	/**
	 * String to hold last error
	 * @var string
	 */
	var $lastError = null;
	/**
	 * Class Constructor
	 * @return AdSync
	 */
	function AdSync($server=null,$port=null,$user=null,$pass=null,$dn=null,$ssl=null,$tls=null) {
		$this->setServer($server);
		$this->setPort($port);
		$this->setUser($user);
		$this->setPass($pass);
		$this->setDN($dn);
		$this->setSSL($ssl);
		$this->setTLS($tls);
	}
	
	/**
	 * Set Server name
	 * @param string $server
	 * @return void
	 */
	function setServer($server) {
		$this->server = $server;
	}
	
	/**
	 * Return the server name
	 * @return string
	 */
	function getServer() {
		if ($this->getSSL()) 
			return 'ldaps://'.$this->server;
		else
			return $this->server;
	}
	
	/**
	 * Set the port
	 * @param int $port Port Number
	 * @return void
	 */
	function setPort($port) {
		$this->port = $port;
	}
	
	/**
	 * Get the port number
	 * @return int
	 */
	function getPort() {
		return $this->port;
	}
	
	/**
	 * Set the user name
	 * @param string $user
	 * @return void
	 */
	function setUser($user) {
		$this->user = $user;
	}
	
	/**
	 * Get the user name
	 * @return string
	 */
	function getUser() {
		return $this->user;
	}
	
	/**
	 * Set the password
	 * @param string $pass
	 * @return void
	 */
	function setPass($pass) {
		$this->pass = $pass;
	}
	
	/**
	 * Get the password for Active Directory
	 * @return string
	 */
	function getPass() {
		return $this->pass;
	}
	
	/**
	 * Set the DN
	 * @param string $dn
	 * @return void
	 */
	function setDN($dn) {
		$this->dn = $dn;
	}
	
	/**
	 * Get the DN
	 * @return string
	 */
	function getDN() {
		$dn = 'DC='.implode(',DC=',explode('.',$this->dn));
		return $dn;
	}
	
	/**
	 * Set the SSL
	 * @param bool $ssl
	 * @return void
	 */
	function setSSL($ssl) {
		$this->ssl = $ssl;
	}
	
	/**
	 * Get the SSL Setting
	 * @return bool
	 */
	function getSSL() {
		return $this->ssl;
	}
	
	/**
	 * Set the TLS Settings
	 * @param bool $tls
	 * @return void
	 */
	function setTLS($tls) {
		$this->tls = $tls;
	}
	
	/**
	 * Get the TLS Settings
	 * @return bool
	 */
	function getTLS() {
		return $this->tls;
	}
	
	/**
	 * Set the search string
	 * @param string $searchString
	 * @return void
	 */
	function setSearchString($searchString) {
		$this->searchString = $searchString;
	}
	
	/**
	 * Return the search string to use
	 * @return string
	 */
	function getSearchString() {
		return $this->searchString;
	}
	
	/**
	 * Set the return fields
	 * @param array $returnFields
	 * @return void
	 */
	function setReturnFields($returnFields) {
		$this->returnFields = $returnFields;
	}
	
	/**
	 * Return the Fields
	 * @return unknown_type
	 */
	function getReturnFields() {
		return $this->returnFields;
	}
	
	/**
	 * Set the last error
	 * @param string $error
	 * @return void
	 */
	function setLastError($error) {
		$this->lastError = $error;
	}

	/**
	 * Get the last error
	 * @return string
	 */
	function getLastError() {
		return $this->lastError;
	}
	
	/**
	 * Connect to AD Server
	 * @return bool
	 */
	function connect() {
		// Attempt Socket connect to ensure port is open
		$fSockPort = ($this->getPort()) ? $this->getPort() : (($this->getSSL()) ? 636 : 389);
		$fp = @fsockopen($this->server,$fSockPort,$errno,$errstr,10);
		if (!$fp) {
			$this->setLastError('Unable to open socket connection to server');
			return false;
		}
		@fclose($fp);
		
		// Check if we need to use a port or not
		if ($this->getPort())
			$this->ad = @ldap_connect($this->getServer(),$this->getPort());
		else
			$this->ad = @ldap_connect($this->getServer());
		
		// Check we have a resource
		if (!$this->ad) {
			$this->setLastError('Unable to connect to server.');
			return false;
		}
		
		// Version
		if (!ldap_set_option($this->ad, LDAP_OPT_PROTOCOL_VERSION, 3)) {
			$this->setLastError('Unable to set protocol version.');
			$this->disconnect();
			return false;
		}
		
		// Referrals
		if (!ldap_set_option($this->ad, LDAP_OPT_REFERRALS,0)) {
			$this->setLastError('Unable to set referrals option.');
			$this->disconnect();
			return false;
		}
		
		// TLS
		if ($this->getTLS()) {
			if (!@ldap_start_tls($this->ad)) {
				$this->setLastError('Unable to start TLS Session with server');
				$this->disconnect();
				return false;
			}
		}
		// Binding to ldap server
		$user = $this->getUser().'@'.$this->dn;
		$bd = @ldap_bind($this->ad, $user, $this->getPass());
		
		if (!$bd) {
			$this->setLastError('Unable to bind to server with supplied username and password');
			$this->disconnect();
			return false;
		}
		
		// return true
		return true;
	}
	
	/**
	 * Disconnect from LDAP server
	 * @return bool
	 */
	function disconnect() {
		if ($this->ad)
			return ldap_unbind($this->ad);
		else
			return true;
	}

	/**
	 * Search in Active Directory
	 * @return array
	 */
	function search() {
		$search = @ldap_search($this->ad, $this->getDN(), $this->getSearchString(), $this->getReturnFields());
		// Ensure we got a correct search connection
		if (!$search) {
			$this->setLastError('Failed to run LDAP Search');
			$this->disconnect();
			return false;
		}
		// return entries
		return ldap_get_entries($this->ad, $search);
		
	}
	
	/**
	 * Search for  email addresses
	 * @return array
	 */
	function searchAddresses($domain) {
		$result = $this->search();
		
		// If there was a problem then this will be false
		if (!$result)
			return false;

		$addresses = array();
		
		// Loop and append addresses to array
		for ($i=0;$i<$result['count'];$i++) {
      if (! array_key_exists ('proxyaddresses', $result[$i]))
        continue;
			for ($y=0;$y<$result[$i]['proxyaddresses']['count'];$y++) {
				if (strpos(strtolower($result[$i]['proxyaddresses'][$y]),'smtp:')===0) {
					$addr = strtolower(substr($result[$i]['proxyaddresses'][$y],5));
          if (strpos ($addr,"@") === false) continue;
					$addresses[] = substr($addr,0,strpos($addr,"@")+1).$domain;
				}
			}
		}
		// return an array without duplicates
		return array_unique($addresses);
	}

}
